<?php

namespace App\Http\Controllers;

use App\Models\Carta;
use App\Models\SubCarta;
use App\Models\Lang;
use App\Models\SubCartaLang;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Log;
use Yajra\DataTables\DataTables;
use Illuminate\Support\Facades\Validator;

class SubCartaController extends Controller
{
    public function index()
    {
        $langs = Lang::whereNull('deleted_at')->get(); // solo idiomas activos
        $cartas = Carta::whereNull('deleted_at')->get(); // solo idiomas activos
        return view('subcartas.index', compact('langs', 'cartas'));
    }

    public function ordenarVista()
    {
        // Traemos las cartas con sus subcartas traducidas
        $cartas = Carta::with([
            'translations',
            'subCartas.translations' => fn($q) => $q
        ])
            ->with(['subCartas' => fn($q) => $q->orderBy('orden')])
            ->orderBy('orden')
            ->get();

        return view('subcartas.ordenar', compact('cartas'));
    }

    public function ordenarGuardar(Request $request)
    {
        $orden = $request->input('orden');
        // $orden será algo como: ['carta_id' => [subcarta1, subcarta2, ...]]

        foreach ($orden as $cartaId => $subCartas) {
            foreach ($subCartas as $pos => $id) {
                SubCarta::where('id', $id)->update(['orden' => $pos]);
            }
        }

        return response()->json(['success' => true]);
    }

    public function datatables(Request $request)
    {
        try {
            // Obtener el código del idioma por defecto (donde default = 1)
            $defaultLocale = Lang::where('default', true)->value('code');

            // Si no hay idioma por defecto configurado, usar 'es' como fallback
            if (!$defaultLocale) {
                $defaultLocale = 'es';

                // Opcional: Intentar corregir automáticamente
                $spanish = Lang::where('code', 'es')->first();
                if ($spanish) {
                    $spanish->default = true;
                    $spanish->save();

                    // Desmarcar otros idiomas como default
                    Lang::where('code', '!=', 'es')->update(['default' => false]);
                }
            }

            $query = SubCarta::leftJoin('sub_carta_langs', function ($join) use ($defaultLocale) {
                $join->on('sub_cartas.id', '=', 'sub_carta_langs.sub_carta_id')
                    ->where('sub_carta_langs.locale', '=', $defaultLocale);
            })
                ->select('sub_cartas.*', 'sub_carta_langs.nombre as nombre_traduccion');

            return DataTables::of($query)
                ->addColumn('nombre', function ($subcarta) use ($defaultLocale) {
                    // Si no hay traducción, mostrar un mensaje
                    if (!$subcarta->nombre_traduccion) {
                        $defaultLang = Lang::where('default', true)->first();
                        $langName = $defaultLang ? $defaultLang->name : 'Default';

                        return '<span class="text-muted">No disponible en ' . $langName . '</span>';
                    }

                    return $subcarta->nombre_traduccion;
                })
                ->editColumn('activo', function ($subcarta) {
                    return $subcarta->activo
                        ? '<span class="badge badge-success">Activo</span>'
                        : '<span class="badge badge-danger">Inactivo</span>';
                })
                ->addColumn('acciones', function ($subcarta) {
                    return '
                <button class="btn btn-sm btn-warning btn-editar" data-id="' . $subcarta->id . '">
                    <i class="fas fa-edit"></i>
                </button>
                <button class="btn btn-sm btn-danger btn-eliminar" data-id="' . $subcarta->id . '">
                    <i class="fas fa-trash"></i>
                </button>
            ';
                })
                ->filterColumn('nombre', function ($query, $keyword) {
                    $query->where('sub_carta_langs.nombre', 'like', "%{$keyword}%");
                })
                ->orderColumn('nombre', 'sub_carta_langs.nombre $1')
                ->rawColumns(['nombre', 'activo', 'acciones'])
                ->make(true);
        } catch (\Exception $e) {
            Log::error('Error en datatables: ' . $e->getMessage());

            return DataTables::of(SubCarta::query())
                ->addColumn('nombre', function ($subcarta) {
                    return '<span class="text-danger">Error al cargar datos</span>';
                })
                ->rawColumns(['nombre'])
                ->make(true);
        }
    }
    public function store(Request $request)
    {
        $validator = Validator::make($request->all(), [
            'carta_id' => 'required|exists:cartas,id',
            'orden' => 'required|integer',
            'activo' => 'boolean',
            'nombre.*' => 'required|string|max:255',
            'descripcion.*' => 'nullable|string',
        ]);

        if ($validator->fails()) {
            return response()->json([
                'success' => false,
                'errors' => $validator->errors()
            ], 422);
        }

        try {
            $subcarta = new SubCarta();
            $subcarta->carta_id = $request->carta_id;
            $subcarta->orden = $request->orden;
            $subcarta->activo = $request->activo ?? true;
            $subcarta->save();

            // Guardar traducciones
            foreach ($request->nombre as $locale => $nombre) {
                $subcarta->translateOrNew($locale)->nombre = $nombre;
                $subcarta->translateOrNew($locale)->descripcion = $request->descripcion[$locale] ?? null;
            }

            $subcarta->save();

            return response()->json([
                'success' => true,
                'message' => 'Subcarta creada correctamente'
            ]);
        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Error al crear la subcarta: ' . $e->getMessage()
            ], 500);
        }
    }

    public function show($id)
    {
        $subCarta = SubCarta::with('translations')->findOrFail($id);

        return response()->json([
            'id' => $subCarta->id,
            'carta_id' => $subCarta->carta_id,
            'orden' => $subCarta->orden,
            'activo' => (bool) $subCarta->activo,
            'translations' => $subCarta->translations->map(function ($t) {
                return [
                    'locale' => $t->locale,
                    'nombre' => $t->nombre,
                    'descripcion' => $t->descripcion,
                ];
            })->values(),
        ]);
    }


    public function update(Request $request, SubCarta $subcarta)
    {
        $validator = Validator::make($request->all(), [
            'carta_id'      => 'required|exists:cartas,id',
            'orden'         => 'required|integer',
            'activo'        => 'boolean',
            'nombre.*'      => 'required|string|max:255',
            'descripcion.*' => 'nullable|string',
        ]);

        if ($validator->fails()) {
            return response()->json([
                'success' => false,
                'errors'  => $validator->errors()
            ], 422);
        }

        try {
            DB::beginTransaction();

            // 🔹 1. Actualizar la subcarta principal
            $subcarta->update([
                'carta_id' => $request->carta_id,
                'orden'    => $request->orden,
                'activo'   => $request->boolean('activo'),
            ]);

            // 🔹 2. Actualizar o crear traducciones
            foreach ($request->nombre as $locale => $nombre) {
                SubCartaLang::updateOrCreate(
                    [
                        'sub_carta_id' => $subcarta->id, // siempre atado al padre
                        'locale'       => $locale,
                    ],
                    [
                        'nombre'      => $nombre,
                        'descripcion' => $request->descripcion[$locale] ?? null,
                    ]
                );
            }

            DB::commit();

            return response()->json([
                'success' => true,
                'message' => 'Subcarta actualizada correctamente',
            ]);
        } catch (\Throwable $e) {
            DB::rollBack();

            return response()->json([
                'success' => false,
                'message' => 'Error al actualizar la subcarta: ' . $e->getMessage(),
            ], 500);
        }
    }





    public function destroy(SubCarta $subCarta)
    {
        try {
            // Soft Delete (no elimina físico, solo marca deleted_at)
            $subCarta->delete();

            return response()->json([
                'success' => true,
                'message' => 'Subcarta eliminada correctamente'
            ]);
        } catch (\Throwable $e) {
            return response()->json([
                'success' => false,
                'message' => 'Error al eliminar la subcarta: ' . $e->getMessage()
            ], 500);
        }
    }
}
